% SpikeSynchrony_SampleBased.m
% 
%   Description: Determines the sychronicity in between two spike between
%       seperate electrode by outputting a coincidence level. Sourced from 
%       Satuvuori et al. 2017 to quantify coincident spikes in two spike trains: 
%       L1 & L2
%
%   Inputs:
%       -L1 and L2: latency vectors are given in terms of sample number with respect
%       to the time vector
%       -time: vector to store time
%       -Maxdiff: threshold 
%
%   Outputs:
%       -C1,C2: coincidence values for the two signals
%       -Deltas: time difference between to events.

function [C1,C2,Deltas] = SpikeSynchrony(L1,L2,time,MaxDiff)

%Variable to keep track of the distance between coincident spikes
Deltas = [];

%Time resolved profile for preceding spikes
L1prev = zeros(size(time));
L2prev = zeros(size(time));

for t = 1:length(time)
    currentL1 = max(L1(L1<=t));
    currentL2 = max(L2(L2<=t));
    if(~isempty(currentL1))
        L1prev(t) = time(currentL1);
    end
    if(~isempty(currentL2))
        L2prev(t) = time(currentL2);
    end
end

%Time resolved profile for following spikes
L1next = time(end)*ones(size(time));
L2next = time(end)*ones(size(time));

for t = 1:length(time)
    currentL1 = min(L1(L1>t));
    currentL2 = min(L2(L2>t));
    if(~isempty(currentL1))
        L1next(t) = time(currentL1);
    end
    if(~isempty(currentL2))
        L2next(t) = time(currentL2);
    end
end

%Inter-spike intervals
X1_ISI = L1next - L1prev;
X2_ISI = L2next - L2prev;
%Do edge correction so that ISI persists before first spike and after last
X1_ISI(1:L1(1)-1) = X1_ISI(L1(1)); X1_ISI(L1(end):end) = X1_ISI(L1(end)-1);
X2_ISI(1:L2(1)-1) = X2_ISI(L2(1)); X2_ISI(L2(end):end) = X2_ISI(L2(end)-1);

%Compute a minimum relevant time scale (MRST) from the data

All_ISI_lengths = [X1_ISI(L1(1:end-1)) X2_ISI(L2(1:end-1))];
thresh = sqrt(sum(All_ISI_lengths.^2)/length(All_ISI_lengths));

%Find coincidence window lengths for each spike in each train
W1 = zeros(size(L1));
W2 = zeros(size(L2));
W1p = zeros(size(L1));
W2p = zeros(size(L2));
W1f = zeros(size(L1));
W2f = zeros(size(L2));

for i = 1:length(L1)
    x1p = X1_ISI(L1(i)-1);
    x1f = X1_ISI(L1(i));
    W1(i) = 0.5*min([x1p x1f]);
    W1p(i) = min([max([0.25*thresh W1(i)]) 0.5*x1p]);
    W1f(i) = min([max([0.25*thresh W1(i)]) 0.5*x1f]);
end

for i = 1:length(L2)
    x2p = X2_ISI(L2(i)-1);
    x2f = X2_ISI(L2(i));
    W2(i) = 0.5*min([x2p x2f]);
    W2p(i) = min([max([0.25*thresh W2(i)]) 0.5*x2p]);
    W2f(i) = min([max([0.25*thresh W2(i)]) 0.5*x2f]);
end

%% Check for coincidences between the two spike trains
C1 = zeros(size(L1));
C2 = zeros(size(L2));

for i = 1:length(L1)
    currentTime = time(L1(i));
    Wij = zeros(size(L2));
    Diffs = zeros(size(L2));
    
    for j = 1:length(L2)
        %compute combined coincidence window for each j spike
        if(L1(i)<=L2(j))
            Wij(j) = min([W1f(i) W2p(j)]);
        else
            Wij(j) = min([W1p(i) W2f(j)]);
        end
        
        Diffs(j) = abs(currentTime - time(L2(j))); 
        
    end
    
    %Decide whether there is a coincidence for i'th spike
    [min_diff,ind] = min(Diffs);
    if(min_diff<Wij(ind))
        %Add in a maximum relevant time difference 
        if(abs(currentTime-time(L2(ind)))<MaxDiff)
            C1(i) = 1;
            C2(ind) = 1;
            Deltas = [Deltas currentTime-time(L2(ind))];
        end
    end
    
end